--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Items
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-- Item class definition
LItem =
{
	__tostring = function(i) return "{Item:"..tostring(i.name).."}" end,
	_ByName = {},
	_ByIndex = {},
	_IngredientsByIndex = {},
	_ProductsByIndex = {},

	truePriceMultiplier = .5,		-- eventual "median" price multiplier between low and high
	lowestMarkup = 1.15,			-- eventual "lowest" markup allowed
	
	startWeek = 0,					-- by default, ingredients are always available
	endWeek = 52,
}

-------------------------------------------------------------------------------
-- Functions for data description

function DefineItem(t)
	t = t or {}
	t.name = t.name or t[1]
	if type(t) == "string" then t = { name=t } end
	return LItem:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global Item lists

function LItem:AllItems()
	return bsgArrayIterator(self._ByIndex)
end

function LItem:ByName(name)
	return self._ByName[name]
end

-- Ingredients
function LItem:AllIngredients()
	return bsgArrayIterator(self._IngredientsByIndex)
end

function LItem:IngredientByIndex(n)
	return self._IngredientsByIndex[n]
end

function LItem:IngredientsByType(t)
	local i = 0
	local n = table.getn(self._IngredientsByIndex)
	local type = t
	return function()
		i = i + 1
		while (i <= n) do
			local ing = self._IngredientsByIndex[i]
			if ing.type == type then return ing end
			i = i + 1
		end
		return nil
	end
end

function LItem:OwnedIngredients()
	local i = 0
	local n = table.getn(self._IngredientsByIndex)
	return function()
		i = i + 1
		while (i <= n) do
			local ing = self._IngredientsByIndex[i]
			if ing.inventory > 0 then return ing end
			i = i + 1
		end
		return nil
	end
end

-- Products
function LItem:AllProducts()
	return bsgArrayIterator(self._ProductsByIndex)
end

function LItem:ProductByIndex(n)
	return self._ProductsByIndex[n]
end

function LItem:OwnedProducts()
	local i = 0
	local n = table.getn(self._ProductsByIndex)
	return function()
		i = i + 1
		while (i <= n) do
			local ing = self._ProductsByIndex[i]
			if ing.inventory > 0 then return ing end
			i = i + 1
		end
		return nil
	end
end

function LItem:KnownRecipes()
	local i = 0
	local n = table.getn(self._ProductsByIndex)
	return function()
		i = i + 1
		while (i <= n) do
			local p = self._ProductsByIndex[i]
			if p.known then return p end
			i = i + 1
		end
		return nil
	end
end

-- Sorting
function ItemSortByDescendingPrice(a,b)
	return b.high < a.high
end

-------------------------------------------------------------------------------
-- "constructor"

function LItem:new(t)
	-- Standard object creation...
	t = t or {} setmetatable(t, self) self.__index = self

	-- Check parameters
	if not t.name then
		bsgDevWarning("Item defined with no name")
	elseif self._ByName[t.name] then
		bsgDevWarning("Item "..t.name.." already defined")
	else
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)
		
		if t.recipe then
			table.insert(self._ProductsByIndex, t)
			t.recipeSize = 0
			t.ingTypeCount = 0
			for ing,count in pairs(t.recipe) do
				t.recipeSize = t.recipeSize + count
				t.ingTypeCount = t.ingTypeCount + 1
			end
		else
			table.insert(self._IngredientsByIndex, t)
		end
		
		t.sold = 0
		t.inventory = 0
		t.usetime = 0
		t.lastprice = 0
		t.lastport = nil
		
		return t
	end
	return nil
end

-------------------------------------------------------------------------------
-- Post-Load Processing

function LItem:PostProcessAll()
	for _,item in ipairs(self._ByIndex) do
		if item.recipe and type(item.type) == "string" then
			-- Keep track of all products in a given type
			local itemType = LProductType:ByName(item.type)
			if not itemType then
				bsgDevWarning("Item "..tostring(item.name).." - undefined type: "..tostring(item.type))
			else
				item.type = itemType
				item.type:AddProduct(item)

				-- Calculate product pricing based on cost of ingredients
				item:CalculateRecipeCosts()
			end
		end
	end
end

function LItem:CalculateRecipeCosts()
	self.lowcost = 0
	self.highcost = 0
	for item,count in pairs(self.recipe) do
		local ing = LItem:ByName(item)
		if not ing then
			bsgDevWarning("Item "..tostring(self.name).." - undefined ingredient: "..tostring(item))
		else
			self.lowcost = self.lowcost + count * ing.low
			self.highcost = self.highcost + count * ing.high
		end
	end
	
	-- calculate standard markup
	local markup = self.markup or self.type.markup or 2
	self.lowest = self.lowcost * self.lowestMarkup
	self.low = self.lowcost * markup
	self.high = self.highcost * markup
	
	-- and "true" price is between low and high
	self.middle = bsgFloor(self.low + (self.high - self.low) * self.truePriceMultiplier + .5)
	self.low = bsgFloor(self.low + .5)
	self.high = bsgFloor(self.high + .5)
end

-------------------------------------------------------------------------------
-- Price randomization and seasonality

function LItem:IsSeasonal()
	if self.startWeek > LItem.startWeek or self.endWeek < LItem.endWeek then return true
	else return false
	end
end

function LItem:InSeason(seasonWeek)
	return (self.startWeek < self.endWeek and (seasonWeek >= self.startWeek and seasonWeek <= self.endWeek)) or
		(self.startWeek > self.endWeek and (seasonWeek >= self.startWeek or seasonWeek <= self.endWeek))
end

function LItem:PreparePrices()
	-- TODO: Redo seasonal product pricing +10%?
	
	local seasonWeek = bsgMod(gSim.weeks, 52)
	for _,item in ipairs(self._ByIndex) do
		local price = bsgRandom(item.low, item.high)
		
		if item.recipe then
			-- Once you're a Master Chocolatier, the number of products you've sold doesn't matter to the product price
			if gSim.rank < 4 then
				-- Apply product pricing rules as product sells
				if item.sold <= 2000 then
					-- Tend towards "middle" price over the first 2000 sales
					price = price + (item.middle - price) * item.sold / 2000
				elseif item.sold < 7000 then
					-- Tend towards the "lowest" price over the next 5000 sales
					price = item.middle + (item.lowest - item.middle) * (item.sold - 2000) / 5000
				elseif item.type.name == "truffle" or item.type.name == "exotic" then
					-- Truffles and Exotics sell at cost forever, rely on "owned shop" markup forever
					price = item.lowcost
				else
					-- After selling 7000, the bottom drops out of the market and everything sells
					-- below cost at non-owned shops
					price = bsgFloor(item.lowcost * .9)
				end
			end
			
			-- Seasonal 10% bump
--			if season > 0 then price = price * 1.1 end
			
			-- 20% markup at owned shops
			-- Once products have dropped to their low price, this will amount to an 8% markup at owned shops
			-- OR a full 20%+ markup on truffles and exotics forever
			if gSim.port.shopOwned then price = price * 1.2 end
			
			price = bsgFloor(price + 0.5)
			if price > item.high then price = item.high end
			if price < 1 then price = 1 end
		else
			-- Apply seasonal pricing to ingredients
			if not item:InSeason(seasonWeek) then
				-- Randomize between 1.5x and 2x high
				-- NOTE: If changing this, be sure to change LMarket:ComputeHaggle to match
				price = bsgRandom(1.5*item.high, 2*item.high)
			end
		end
		
		item.price = price
	end
	
	if gSim.rank == 0 then
		-- Tutorial pricing on sugar and cacao
		local item = LItem:ByName("sugar")
--		if item then item.price = bsgFloor((item.high - item.low) / 5) + item.low end
		if item then item.price = 30 end
		item = LItem:ByName("cacao")
--		if item then item.price = bsgFloor((item.high - item.low) / 5) + item.low end
		if item then item.price = 75 end
	end
end

-------------------------------------------------------------------------------
-- Recipe management

function LItem:EnableRecipe(reason)
	if not self.recipe then
		DebugOut("Item "..tostring(self.name).." has no recipe")
	elseif not self.known then
		-- Open recipe book to this page next
		gRecipeSelection = self
		self.known = true
		self.newrecipe = true
		self.type:IncrementKnownCount()
		if reason then
			-- Ledger will be updated by whoever called this
			gSim:QueueMessage(GetString(reason, GetString(self.name)))
		end
	end
end

-------------------------------------------------------------------------------
-- Popups

function LItem:Rollover(x,y,scale,clicklabel,command)
	local imageName = "item/"..self.name
	if self.lab and (not self.known) then imageName = "item/unknown" end
	return Rollover { x=x,y=y,
		contents=self:RolloverTarget(clicklabel),
		scale=scale, command=command,
		Bitmap { image=imageName, scale=scale } }
end

function LItem:BigRollover(x,y,scale,clicklabel,command)
	local imageName = "item/"..self.name.."_big"
	if self.lab and (not self.known) then imageName = "item/unknown_big" end
	return Rollover { x=x,y=y,
		contents=self:RolloverTarget(clicklabel),
		scale=scale, command=command,
		Bitmap { image=imageName, scale=scale } }
end

function LItem:RolloverTarget(clicklabel)
	if clicklabel then clicklabel = "'"..tostring(clicklabel).."'" end
	if self.recipe then return "(LItem:ByName('"..tostring(self.name).."')):ProductRollover("..tostring(clicklabel)..")"
	else return "(LItem:ByName('"..tostring(self.name).."')):IngredientRollover("..tostring(clicklabel)..")"
	end
end

function LItem:IngredientRollover(clicklabel)
	local font = popupFont
	local h = bsgFontHeight(font)
	local y = h

	local text = "#"..GetString(self.name)
	if self:IsSeasonal() then
		local seasonWeek = bsgMod(gSim.weeks, 52)
		if self:InSeason(seasonWeek) then text = text .. ": " .. GetString("in_season")
		else text = text .. ": " .. GetString("out_season")
		end
	end
	text = text.." ("..GetString("ing_inventory", tostring(self.inventory))..")"	
	
	if self.lastport and self.lastprice > 0 then
		local l
		if self.inventory > 0 then
			l = GetString("last_bought", GetString(self.lastport), bsgDollars(self.lastprice))
		else
			l = GetString("last_seen", GetString(self.lastport), bsgDollars(self.lastprice))
		end
		text = text.."<br>"..l
		y = y + h
	end
	
	local low = self.lowseen or self.highseen
	local high = self.highseen or low
	if low then
		if low < high then text = text .. "<br>" .. GetString("price_range", bsgDollars(low), bsgDollars(high))
		else text = text .. "<br>" .. GetString("lowest_seen", bsgDollars(low))
		end
		y = y + h
	end
	
	-- Add optional additional text
	if clicklabel then
		y = y + h
		text = text .. "<br>" .. GetString(clicklabel)
	end
	
	-- Center on item, or on text if higher than item...
	y = (2*ui.itemHeight - y)/2
	local iy = 0,0
	if y < 0 then
		iy = -y
		y = 0
	end
	
	return MakeRollover
	{
		x=0,y=0, color=PopupColor, inset=2, alpha=0.8,
		Bitmap { x=0,y=iy,w=2*ui.itemWidth,h=2*ui.itemHeight, image="item/"..self.name.."_big" },
		TightText { x=2*ui.itemWidth+2,y=y,h=2*ui.itemHeight, label=text, font=font, flags=kHAlignLeft+kVAlignTop },
	}
end

function LItem:ProductRollover(clicklabel)
	local font = popupFont
	local h = bsgFontHeight(font)
	local y = h
	
	local name = self.name
	if self.lab and not self.known then name = "lab_unknown_recipe" end
	
	local text = "#"..GetString(name)
	if self.known then
		if self.sold > 0 then
			local s = GetString("item_sold", tostring(self.sold))
			if self.inventory > 0 then s = s..", "..GetString("prod_inventory", tostring(self.inventory)) end
			text = text.."<br>"..s
			y = y + h
		elseif self.inventory > 0 then
			local s = GetString("prod_inventory", tostring(self.inventory))
			text = text.."<br>"..s
			y = y + h
		end

		if self.lastport and self.lastprice > 0 then
			local l = GetString("last_sold", GetString(self.lastport), bsgDollars(self.lastprice))
			text = text.."<br>"..l
			y = y + h
		end

		local low = self.lowseen or self.highseen
		local high = self.highseen or low
		if low then
			if low < high then text = text .. "<br>" .. GetString("price_range", bsgDollars(low), bsgDollars(high))
			else text = text .. "<br>" .. GetString("highest_seen", bsgDollars(high))
			end
			y = y + h
		end
	end
	
	-- Add optional additional text
	if clicklabel then
		y = y + h
		text = text .. "<br>" .. GetString(clicklabel)
	end
	
	-- Center on item, or on text if higher than item...
	y = (2*ui.itemHeight - y)/2
	local iy = 0
	if y < 0 then
		iy = -y
		y = 0
	end

	local imageName = "item/"..self.name.."_big"
	if self.lab and (not self.known) then imageName="item/unknown_big" end
	return MakeRollover
	{
		x=0,y=0, color=PopupColor, inset=2, alpha=.8,
		Bitmap { x=0,y=iy,w=2*ui.itemWidth,h=2*ui.itemHeight, image=imageName },
		TightText { x=2*ui.itemWidth+2,y=y,h=2*ui.itemHeight, label=text, font=font, flags=kHAlignLeft+kVAlignTop },
	}
end

-------------------------------------------------------------------------------
-- Reset

function LItem:ResetAll()
	for _,i in ipairs(self._ByIndex) do
		i.known = nil
		i.sold = 0
		i.inventory = 0
		i.lastprice = 0
		i.lastport = nil
		i.usetime = 0
		i.lowseen = nil
		i.highseen = nil
		i.newrecipe = nil
	end
end

-------------------------------------------------------------------------------
-- Load and Save

function LItem:BuildSaveTable()
	local t = {}
	for _,item in ipairs(self._ByIndex) do
		local it =
		{
			price = item.price,
			known = item.known,
			lowseen = item.lowseen,
			highseen = item.highseen,
			newrecipe = item.newrecipe,
		}
		
		if item.inventory > 0 then it.inventory = item.inventory end
		if item.sold > 0 then it.sold = item.sold end
		if item.usetime > 0 then it.usetime = item.usetime end
		
		if item.lastport then
			it.lastprice = item.lastprice
			it.lastport = item.lastport
		end
		
		t[item.name] = it
	end
	return t
end

function LItem:LoadSaveTable(t)
	for name,data in pairs(t) do
		local item = self:ByName(name)
		if item then
			item.price = data.price
			item.sold = data.sold or 0
			item.inventory = data.inventory or 0
			item.usetime = data.usetime or 0
			item.known = data.known
			item.lowseen = data.lowseen
			item.highseen = data.highseen
			item.newrecipe = data.newrecipe
			
			if data.lastport then
				item.lastprice = data.lastprice
				item.lastport = data.lastport
			end
		end
	end
end
